import os
import sys
from typing import List, Any

import requests
from mcp.server.fastmcp import FastMCP

sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), "../")))
from config.system import config

mcp = FastMCP("BME-MCP", log_level="ERROR")

remote = config["remote"]

headers = remote["headers"]

base_url = remote["base_url"]

bme_screen_service = base_url["bme-screen-service"]


def deal_request_exception(response):
    """
    处理response的异常信息
    :param response: 响应体
    :return: json数据
    """
    response.raise_for_status()
    response_json = response.json()
    if isinstance(response_json, dict) and "success" in response_json and not response_json["success"]:
        raise Exception(response_json.get("msg"))
    return response_json


def get_size_limit(size: int, max_size=20, min_size=0) -> int:
    """
    处理数据量边界
    :param size: 数据量
    :param max_size: 最大边界
    :param min_size: 最小边界
    :return:
    """
    size = max_size if size > max_size else size
    size = min_size if size < min_size else size
    return size


@mcp.tool()
def get_air_pm_10_month_focus(customer_id: int) -> Any:
    """
    根据客户id获取本月重点关注的空气质量微站（PM10标准）
    :param customer_id: 客户id
    :return: 返回结构中的字段名解释：
        rank：排行,
        deviceNo：设备编号，
        deviceName：设备名称
    """
    response = requests.get(bme_screen_service + "/monitor/getMonitorRankingTitle", {
        "customerId": customer_id,
        "type": "1",  # 随便传，该参数当前无用
        "order": "1",  # 随便传，该参数当前无用
    }, headers=headers)
    response_data = deal_request_exception(response).get("data", {}).get("title", [])
    return [{
        "rank": data.get("rank"),
        "deviceNo": data.get("deviceNo"),
        "deviceName": data.get("deviceName")
    } for data in response_data]


@mcp.tool()
def get_air_pm_rank(customer_id: int, order: int, statistic_type: int) -> Any:
    """
    根据客户id获取空气质量微站排行（包含PM2.5和PM10标准），其中
    order如果没有指定，默认值是1;
    statistic_type如果没有指定，默认值是1;

    :param customer_id: 客户id
    :param order: 排序规则：按统计区间的类型进行排序，升序排序传1，降序排序传2，默认为1
    :param statistic_type: 统计区间：按小时统计传1，按日统计传2，要按月统计传3，默认为1
    :return: 返回结构中的字段名解释：
        PM10：PM10的数据集，
        PM2.5：PM2.5的数据集，
        avgValue：平均值 单位为μg/m³，
        deviceNo：设备编号，
        deviceName：设备名称，
    """
    response = requests.get(bme_screen_service + "/monitor/getMonitorRanking", {
        "customerId": customer_id,
        "type": statistic_type,
        "order": order,
    }, headers=headers)
    response_data = deal_request_exception(response).get("data", {})
    result = {
        "PM10": [{
            "avgValue": pm10.get("avg_value"),
            "deviceNo": pm10.get("deviceNo"),
            "deviceName": pm10.get("deviceName")
        } for pm10 in response_data.get("pm10lists")],
        "PM2.5": [{
            "avgValue": pm2_5.get("avg_value"),
            "deviceNo": pm2_5.get("deviceNo"),
            "deviceName": pm2_5.get("deviceName")
        } for pm2_5 in response_data.get("pm25lists")],
    }
    return result


@mcp.tool()
def get_tsp_rank(customer_id: int, order: int, statistic_type: int, size: int) -> Any:
    """
    根据客户id获取TSP监测排行，其中
    order如果没有指定，默认值是1;
    statistic_type如果没有指定，默认值是1;
    size如果没有指定，默认值是10;

    :param customer_id: 客户id
    :param order: 排序规则：按统计区间的类型进行排序，升序排序传1，降序排序传2，默认为1
    :param statistic_type: 统计区间：按小时统计传1，按日统计传2，要按月统计传3，默认为1
    :param size: 要取多少条数据，比如按小时升序取前两条，则传2，默认为10
    :return: 返回结构中的字段名解释：
            total: 区域总数
            data: 详细数据
            data->area: 区域名称
            data->devicecount: 关联设备数
            data->realtimePollutionIndex: 实时污染指数
            data->hourRank: 小时排名
            data->dayRank: 日排名
            data->monthRank: 月排名
    """
    response = requests.get(bme_screen_service + "/monitor/getMonitorTSPRanking", {
        "customerId": customer_id,
        "order": order,
        "pageNo": 1,
        "pageSize": get_size_limit(size),
        "type": statistic_type,
    }, headers=headers)
    response_data = deal_request_exception(response).get("data", {})
    result = {
        "total": response_data.get("areaCount", 0),
        "data": [{
            "area": data.get("area"),
            "devicecount": data.get("devicecount"),
            "realtimePollutionIndex": f"{data.get('tdc')} {data.get('tspUnit')}",
            "hourRank": data.get("hourRank"),
            "dayRank": data.get("dayRank"),
            "monthRank": data.get("monthRank"),
        } for data in response_data.get("page", {}).get("records")],
    }
    return result


@mcp.tool()
def get_governance_process_statistics(customer_id: int) -> Any:
    """
    根据客户id获取今天治理过程记录统计数据
    :param customer_id: 客户id
    :return: 返回结构中的字段名解释：
        timeCount: 今日雾炮平均开启时长（小时）
        allCount: 今日污染发生次数（次）
        openCount: 今日雾炮开启次数（次）
    """
    response = requests.get(bme_screen_service + "/eq/selectEquCount", {"customerId": customer_id}, headers=headers)
    response_data = deal_request_exception(response).get("data", {})
    result = {
        "timeCount": response_data.get("timeCount", 0),
        "allCount": response_data.get("allCount", 0),
        "openCount": response_data.get("openCount", 0),
    }
    return result


@mcp.tool()
def get_governance_process_records(customer_id: int, instruct_type: str, device_name: str,
                                   start_time: str, end_time: str, size: int) -> Any:
    """
    根据客户id查询治理过程全记录，其中：
    instruct_type如果没有指定，默认值是空字符串;
    device_name如果没有指定，默认值是空字符串;
    start_time如果没有指定，默认值是当前时间的七天前，如：2025-04-23 16:18:00;
    end_time如果没有指定，默认值是当前时间，如：2025-04-30 16:18:00;
    size如果没有指定，默认值是10;

    :param customer_id: 客户id
    :param instruct_type: 触发机制：当传入智能联动时，转换为0；当传入手动开启时，转换为1；其他值都转换为空字符串
    :param device_name: 设备名称
    :param start_time: 开始时间，格式为yyyy-MM-dd HH:mm:ss
    :param end_time: 结束时间，格式为yyyy-MM-dd HH:mm:ss
    :param size: 查询数量，默认10
    :return: 返回结构中的字段名解释：
        total：总数（由于该接口是分页查询，所以这里返回总数）
        data：数据集
        data->createTime: 时间
        data->deviceName: 设备
        data->position: 位置
        data->instructType: 触发机制，0为智能联动，1为手动开启
        data->describe: 治理过程描述
    """
    response = requests.get(bme_screen_service + "/eq/selectEquPage", {
        "customerId": customer_id,
        "InstructType": instruct_type,
        "deviceName": device_name,
        "StatcDate": start_time,
        "StopDate": end_time,
        "pageNo": 1,
        "pageSize": get_size_limit(size),
    }, headers=headers)
    response_data = deal_request_exception(response).get("data", {})
    result = {
        "total": response_data.get("total", 0),
        "data": [{
            "createTime": data.get("createTime"),
            "deviceName": data.get("deviceName"),
            "position": data.get("position"),
            "instructType": data.get("instructType"),
            "describe": data.get("describe"),
        } for data in response_data.get("records")],
    }
    return result


@mcp.tool()
def get_stopped_dust_collector(customer_id: int) -> Any:
    """
    根据客户id获取当前已停止的除尘器
    :param customer_id: 客户id
    :return: 返回结构中的字段名解释：
        name: 所在区域
        deviceName: 除尘器名称
        deviceNo: 除尘器编码
        type: 除尘器类型
    """
    response = requests.get(bme_screen_service + "/treatment/cc_list", {
        "customerId": customer_id,
        "pageNo": 1,
        "pageSize": 50,
        "status": 0
    }, headers=headers)
    response_data = deal_request_exception(response).get("data", {}).get("records", [])
    result = [{
        "name": record.get("name"),
        "deviceName": record.get("deviceName"),
        "deviceNo": record.get("deviceNo"),
        "type": record.get("type")
    } for record in response_data]
    return result


@mcp.resource("api://customers")
def get_all_available_customer() -> Any:
    """
    获取所有的客户信息
    :return: 客户信息数据
    """
    response = requests.get(bme_screen_service + "/reportgen/customer", headers=headers)
    response_data = deal_request_exception(response).get("data", [])
    return [{
        "customerId": data.get("customerId"),
        "customerName": data.get("customerName"),
        "customerFullname": data.get("customerFullname")
    } for data in response_data]


@mcp.prompt(name="default_system_prompt")
def get_default_system_prompt() -> str:
    """
    默认的系统提示词
    :return: 默认的系统提示词
    """
    return (
        "你可以结合一系列的工具（tool）来回答用户的问题。\n"
        "以下是你应该始终遵循的规则：\n"
        "1.始终传入类型正确的参数，如果从用户输入中没有解析到参数，直接使用描述中提到的参数默认值，不需要询问用户。\n"
        "2.只在需要时调用工具，如果你不需要额外信息，不要调用搜索代理，尽量自己解决任务。\n"
        "3.如果不需要调用工具，直接回答问题即可。\n"
        "4.永远不要用完全相同的参数重新进行之前的工具调用。\n"
        "5.如果要使用到客户信息，当用名字查找时，优先匹配客户全称，其次匹配客户名，再考虑使用部分匹配，最后考虑读音相近的名字，都没找到则结束流程并告知用户客户不存在。\n"
    )


if __name__ == '__main__':
    mcp.run(transport="stdio")
