/// <reference types="cypress" />

/**
 * 首页API接口正确性测试
 * 测试首页相关的四个主要API接口的正确性
 */

describe('首页API接口正确性测试', () => {
  let authToken;
  
  before(() => {
    // 获取认证令牌
    cy.mockLogin().then(() => {
      authToken = window.localStorage.getItem('TOKEN');
    });
  });

  beforeEach(() => {
    // 设置通用的请求头
    cy.intercept('GET', '**/api/home/**', (req) => {
      if (authToken) {
        req.headers['TOKEN'] = authToken;
      }
    });
  });

  describe('健康度概览接口测试', () => {
    it('应该成功获取健康度概览数据', () => {
      cy.request({
        method: 'GET',
        url: '/api/home/health/overview',
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        // 验证响应状态码
        expect(response.status).to.equal(200);
        
        // 验证响应体结构
        expect(response.body).to.have.property('code');
        expect(response.body).to.have.property('data');
        expect(response.body).to.have.property('message');
        
        // 验证成功响应码
        expect(response.body.code).to.equal(200);
        
        // 验证数据结构（如果有数据）
        if (response.body.data) {
          const data = response.body.data;
          // 健康度数据应该包含数值类型的字段
          expect(data).to.be.an('object');
          
          // 验证可能的健康度字段
          if (data.average !== undefined) {
            expect(data.average).to.be.a('number');
            expect(data.average).to.be.at.least(0);
            expect(data.average).to.be.at.most(100);
          }
          
          if (data.bag !== undefined) {
            expect(data.bag).to.be.a('number');
            expect(data.bag).to.be.at.least(0);
            expect(data.bag).to.be.at.most(100);
          }
          
          if (data.pulseValve !== undefined) {
            expect(data.pulseValve).to.be.a('number');
            expect(data.pulseValve).to.be.at.least(0);
            expect(data.pulseValve).to.be.at.most(100);
          }
          
          if (data.poppetValve !== undefined) {
            expect(data.poppetValve).to.be.a('number');
            expect(data.poppetValve).to.be.at.least(0);
            expect(data.poppetValve).to.be.at.most(100);
          }
        }
        
        // 验证响应时间
        expect(response.duration).to.be.lessThan(5000);
      });
    });

    it('应该处理健康度概览接口的错误响应', () => {
      // 模拟服务器错误
      cy.intercept('GET', '**/api/home/health/overview', {
        statusCode: 500,
        body: {
          code: 500,
          message: '服务器内部错误',
          data: null
        }
      }).as('healthOverviewError');

      cy.request({
        method: 'GET',
        url: '/api/home/health/overview',
        failOnStatusCode: false,
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        expect(response.status).to.equal(500);
        expect(response.body.code).to.equal(500);
        expect(response.body.message).to.contain('错误');
      });
    });
  });

  describe('健康度统计接口测试', () => {
    it('应该成功获取健康度统计数据', () => {
      cy.request({
        method: 'GET',
        url: '/api/home/health/statistic',
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        // 验证响应状态码
        expect(response.status).to.equal(200);
        
        // 验证响应体结构
        expect(response.body).to.have.property('code');
        expect(response.body).to.have.property('data');
        expect(response.body).to.have.property('message');
        
        // 验证成功响应码
        expect(response.body.code).to.equal(200);
        
        // 验证图表数据结构
        if (response.body.data) {
          const data = response.body.data;
          expect(data).to.be.an('object');
          
          // 验证可能的图表数据字段
          if (data.xAxis) {
            expect(data.xAxis).to.be.an('array');
          }
          
          if (data.series) {
            expect(data.series).to.be.an('array');
          }
          
          if (data.categories) {
            expect(data.categories).to.be.an('array');
          }
        }
        
        // 验证响应时间
        expect(response.duration).to.be.lessThan(5000);
      });
    });

    it('应该处理健康度统计接口的超时', () => {
      // 模拟超时
      cy.intercept('GET', '**/api/home/health/statistic', (req) => {
        req.reply((res) => {
          return new Promise((resolve) => {
            setTimeout(() => resolve(res.send({ statusCode: 408 })), 6000);
          });
        });
      }).as('healthStatisticTimeout');

      cy.request({
        method: 'GET',
        url: '/api/home/health/statistic',
        timeout: 5000,
        failOnStatusCode: false,
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        // 应该处理超时情况
        expect([408, 504]).to.include(response.status);
      });
    });
  });

  describe('异常监控接口测试', () => {
    it('应该成功获取异常监控数据', () => {
      cy.request({
        method: 'GET',
        url: '/api/home/exception/monitor',
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        // 验证响应状态码
        expect(response.status).to.equal(200);
        
        // 验证响应体结构
        expect(response.body).to.have.property('code');
        expect(response.body).to.have.property('data');
        expect(response.body).to.have.property('message');
        
        // 验证成功响应码
        expect(response.body.code).to.equal(200);
        
        // 验证异常监控数据结构
        if (response.body.data) {
          const data = response.body.data;
          
          if (Array.isArray(data)) {
            // 如果是数组，验证数组项结构
            data.forEach(item => {
              expect(item).to.be.an('object');
              
              // 验证可能的异常监控字段
              if (item.time) {
                expect(item.time).to.be.a('string');
              }
              
              if (item.level) {
                expect(item.level).to.be.a('string');
              }
              
              if (item.message) {
                expect(item.message).to.be.a('string');
              }
            });
          } else {
            expect(data).to.be.an('object');
          }
        }
        
        // 验证响应时间
        expect(response.duration).to.be.lessThan(5000);
      });
    });

    it('应该验证异常监控数据的完整性', () => {
      cy.request({
        method: 'GET',
        url: '/api/home/exception/monitor',
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        expect(response.status).to.equal(200);
        
        if (response.body.data && Array.isArray(response.body.data)) {
          const data = response.body.data;
          
          // 验证数据量合理性
          expect(data.length).to.be.at.most(1000); // 单次返回不超过1000条
          
          // 验证时间戳格式（如果存在）
          data.forEach(item => {
            if (item.timestamp) {
              expect(new Date(item.timestamp).getTime()).to.be.a('number');
            }
            
            if (item.createTime) {
              expect(new Date(item.createTime).getTime()).to.be.a('number');
            }
          });
        }
      });
    });
  });

  describe('除尘器告警接口测试', () => {
    it('应该成功获取除尘器告警数据', () => {
      cy.request({
        method: 'GET',
        url: '/api/home/duster/alarm',
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        // 验证响应状态码
        expect(response.status).to.equal(200);
        
        // 验证响应体结构
        expect(response.body).to.have.property('code');
        expect(response.body).to.have.property('data');
        expect(response.body).to.have.property('message');
        
        // 验证成功响应码
        expect(response.body.code).to.equal(200);
        
        // 验证告警数据结构
        if (response.body.data) {
          const data = response.body.data;
          
          if (Array.isArray(data)) {
            // 如果是数组，验证数组项结构
            data.forEach(item => {
              expect(item).to.be.an('object');
              
              // 验证可能的告警字段
              if (item.alarmLevel) {
                expect(item.alarmLevel).to.be.a('string');
              }
              
              if (item.alarmType) {
                expect(item.alarmType).to.be.a('string');
              }
              
              if (item.deviceName) {
                expect(item.deviceName).to.be.a('string');
              }
              
              if (item.location) {
                expect(item.location).to.be.an('object');
              }
            });
          } else {
            expect(data).to.be.an('object');
          }
        }
        
        // 验证响应时间
        expect(response.duration).to.be.lessThan(5000);
      });
    });

    it('应该验证告警数据的地理位置信息', () => {
      cy.request({
        method: 'GET',
        url: '/api/home/duster/alarm',
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        expect(response.status).to.equal(200);
        
        if (response.body.data && Array.isArray(response.body.data)) {
          const data = response.body.data;
          
          data.forEach(item => {
            // 验证地理位置信息
            if (item.location) {
              const location = item.location;
              
              if (location.latitude) {
                expect(location.latitude).to.be.a('number');
                expect(location.latitude).to.be.at.least(-90);
                expect(location.latitude).to.be.at.most(90);
              }
              
              if (location.longitude) {
                expect(location.longitude).to.be.a('number');
                expect(location.longitude).to.be.at.least(-180);
                expect(location.longitude).to.be.at.most(180);
              }
            }
          });
        }
      });
    });
  });

  describe('接口并发访问测试', () => {
    it('应该能够同时访问所有首页接口', () => {
      const requests = [
        cy.request({
          method: 'GET',
          url: '/api/home/health/overview',
          headers: { 'TOKEN': authToken, 'Content-Type': 'application/json' }
        }),
        cy.request({
          method: 'GET',
          url: '/api/home/health/statistic',
          headers: { 'TOKEN': authToken, 'Content-Type': 'application/json' }
        }),
        cy.request({
          method: 'GET',
          url: '/api/home/exception/monitor',
          headers: { 'TOKEN': authToken, 'Content-Type': 'application/json' }
        }),
        cy.request({
          method: 'GET',
          url: '/api/home/duster/alarm',
          headers: { 'TOKEN': authToken, 'Content-Type': 'application/json' }
        })
      ];

      // 并发执行所有请求
      Cypress.Promise.all(requests).then((responses) => {
        // 验证所有请求都成功
        responses.forEach((response, index) => {
          expect(response.status).to.equal(200);
          expect(response.body.code).to.equal(200);
          
          // 验证响应时间合理
          expect(response.duration).to.be.lessThan(10000);
        });
        
        // 验证总体响应时间
        const totalTime = responses.reduce((sum, resp) => sum + resp.duration, 0);
        expect(totalTime).to.be.lessThan(20000); // 总时间不超过20秒
      });
    });
  });

  describe('接口错误处理测试', () => {
    it('应该正确处理无效的TOKEN', () => {
      cy.request({
        method: 'GET',
        url: '/api/home/health/overview',
        failOnStatusCode: false,
        headers: {
          'TOKEN': 'invalid_token_12345',
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        // 应该返回认证错误
        expect([401, 403]).to.include(response.status);
      });
    });

    it('应该正确处理缺少TOKEN的请求', () => {
      cy.request({
        method: 'GET',
        url: '/api/home/health/overview',
        failOnStatusCode: false,
        headers: {
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        // 应该返回认证错误
        expect([401, 403]).to.include(response.status);
      });
    });

    it('应该正确处理网络错误', () => {
      // 模拟网络错误
      cy.intercept('GET', '**/api/home/health/overview', {
        forceNetworkError: true
      }).as('networkError');

      cy.request({
        method: 'GET',
        url: '/api/home/health/overview',
        failOnStatusCode: false,
        retryOnNetworkFailure: false,
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        // 网络错误应该被正确处理
        expect(response).to.exist;
      }).catch((error) => {
        // 或者抛出网络错误异常
        expect(error.message).to.contain('network');
      });
    });
  });

  describe('接口数据一致性测试', () => {
    it('应该验证多次调用返回数据的一致性', () => {
      let firstResponse;
      
      // 第一次调用
      cy.request({
        method: 'GET',
        url: '/api/home/health/overview',
        headers: {
          'TOKEN': authToken,
          'Content-Type': 'application/json'
        }
      }).then((response) => {
        firstResponse = response.body;
        expect(response.status).to.equal(200);
      }).then(() => {
        // 短时间内第二次调用
        cy.request({
          method: 'GET',
          url: '/api/home/health/overview',
          headers: {
            'TOKEN': authToken,
            'Content-Type': 'application/json'
          }
        }).then((response) => {
          expect(response.status).to.equal(200);
          
          // 验证数据结构一致性
          expect(response.body).to.have.property('code');
          expect(response.body).to.have.property('data');
          expect(response.body).to.have.property('message');
          
          // 验证数据类型一致性
          if (firstResponse.data && response.body.data) {
            expect(typeof response.body.data).to.equal(typeof firstResponse.data);
          }
        });
      });
    });
  });
});