/// <reference types="cypress" />

describe('仪表板功能测试', () => {
  beforeEach(() => {
    // 模拟登录状态
    cy.mockLogin()
    
    // 等待登录状态设置完成
    cy.wait(500)
    
    // 访问仪表板
    cy.visit('/#/dashboard')
    
    // 等待页面加载完成
    cy.waitForPageLoad()
  })

  it('应该显示仪表板的所有核心组件', () => {
    // 检查主容器
    cy.get('[data-testid="dashboard-container"]').should('be.visible')
    
    // 检查头部区域
    cy.get('[data-testid="dashboard-header"]').should('be.visible')
    
    // 检查消息框
    cy.get('[data-testid="dashboard-msg-box"]').should('be.visible')
    cy.get('[data-testid="dashboard-msg-item"]').should('be.visible')
    
    // 检查指标框
    cy.get('[data-testid="dashboard-indicators-box"]').should('be.visible')
    cy.get('[data-testid="dashboard-health-score"]').should('be.visible')
    
    // 检查图表区域
    cy.get('[data-testid="dashboard-chart-box"]').should('be.visible')
    cy.get('[data-testid="dashboard-chart-line"]').should('be.visible')
    
    // 检查地图区域
    cy.get('[data-testid="dashboard-map-box"]').should('be.visible')
    cy.get('[data-testid="dashboard-map-svg"]').should('be.visible')
  })

  it('应该显示健康度指标', () => {
    cy.verifyHealthIndicators()
    
    // 检查健康度数值格式
    cy.get('[data-testid="dashboard-health-score"]')
      .should('be.visible')
      .and('contain', '%')
    
    // 检查进度条
    cy.get('[data-testid="dashboard-bag-progress"]').should('be.visible')
    cy.get('[data-testid="dashboard-pulse-valve-progress"]').should('be.visible')
    cy.get('[data-testid="dashboard-poppet-valve-progress"]').should('be.visible')
  })

  it('应该加载并显示图表数据', () => {
    // 等待图表组件加载
    cy.get('[data-testid="dashboard-chart-line"]').should('be.visible')
    
    // 检查图表是否有内容（这里需要根据实际图表实现调整）
    cy.get('[data-testid="dashboard-chart-line"]').within(() => {
      // 检查图表容器内是否有内容
      cy.get('*').should('have.length.gt', 0)
    })
  })

  it('应该显示地图组件', () => {
    // 检查地图组件
    cy.get('[data-testid="dashboard-map-svg"]').should('be.visible')
    
    // 检查地图是否有内容
    cy.get('[data-testid="dashboard-map-svg"]').within(() => {
      cy.get('*').should('have.length.gt', 0)
    })
  })

  it('应该响应数据更新', () => {
    // 记录初始健康度值
    cy.get('[data-testid="dashboard-health-score"]').then(($el) => {
      const initialValue = $el.text()
      
      // 等待一段时间，检查数据是否可能更新
      cy.wait(2000)
      
      // 验证元素仍然存在并且可能有更新
      cy.get('[data-testid="dashboard-health-score"]').should('be.visible')
    })
  })

  it('应该处理消息列表', () => {
    // 检查消息组件
    cy.get('[data-testid="dashboard-msg-item"]').should('be.visible')
    
    // 如果有消息项，检查其结构
    cy.get('[data-testid="dashboard-msg-item"]').within(() => {
      // 检查是否有消息内容
      cy.get('*').should('have.length.gte', 0)
    })
  })

  it('应该验证布局响应式', () => {
    // 测试不同屏幕尺寸下的布局
    const viewports = [
      { width: 1920, height: 1080 },
      { width: 1280, height: 720 },
      { width: 1024, height: 768 }
    ]
    
    viewports.forEach(viewport => {
      cy.viewport(viewport.width, viewport.height)
      cy.wait(500)
      
      // 验证主要组件在不同尺寸下仍然可见
      cy.get('[data-testid="dashboard-container"]').should('be.visible')
      cy.get('[data-testid="dashboard-header"]').should('be.visible')
      cy.get('[data-testid="dashboard-map-box"]').should('be.visible')
    })
  })

  it('应该验证颜色主题', () => {
    // 检查健康度指标的颜色是否根据数值变化
    cy.get('[data-testid="dashboard-health-score"]').then(($el) => {
      const healthScore = parseInt($el.text().replace('%', ''))
      
      // 根据健康度检查颜色
      if (healthScore >= 90) {
        cy.get('[data-testid="dashboard-health-score"]')
          .should('have.css', 'color')
          .and('not.equal', 'rgba(0, 0, 0, 0)')
      }
    })
  })

  it('应该处理数据加载状态', () => {
    // 重新加载页面检查加载状态
    cy.reload()
    
    // 等待页面完全加载
    cy.waitForPageLoad()
    
    // 验证所有关键元素都已加载
    cy.get('[data-testid="dashboard-container"]').should('be.visible')
    cy.get('[data-testid="dashboard-health-score"]').should('be.visible')
  })

  it('应该正确处理无权限用户的重定向', () => {
    // 清除localStorage模拟无权限状态
    cy.window().then((win) => {
      win.localStorage.clear()
      win.sessionStorage.clear()
      // 清除所有cookie
      cy.clearCookies()
    })
    
    // 尝试访问仪表板，应该被重定向到登录页
    cy.visit('/#/dashboard')
    cy.url().should('include', '/#/login')
  })
})