#!/bin/bash

# DC-TOM 测试报告清理脚本
# 专门用于清理历史测试报告、视频和截图文件
# 默认保留最近3次测试结果

set -e

# 颜色输出
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 配置
DEFAULT_KEEP_COUNT=3
REPORTS_DIR="cypress/reports"

# 日志函数
log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 显示使用说明
show_usage() {
    echo "用法: $0 [保留数量]"
    echo ""
    echo "参数:"
    echo "  保留数量    可选，指定保留最近几次测试结果 (默认: $DEFAULT_KEEP_COUNT)"
    echo ""
    echo "示例:"
    echo "  $0        # 保留最近 $DEFAULT_KEEP_COUNT 次测试结果"
    echo "  $0 5      # 保留最近 5 次测试结果"
    echo "  $0 1      # 仅保留最新的测试结果"
    echo ""
    echo "清理内容:"
    echo "  - cypress/reports/ 目录下的 HTML 和 JSON 报告文件"
    echo "  - cypress/videos/ 目录下的测试执行视频"
    echo "  - cypress/screenshots/ 目录下的失败截图"
    echo "  - public/reports/ 目录下的合并报告（可选）"
    echo ""
    echo "注意事项:"
    echo "  - 此脚本会根据文件修改时间排序，保留最新的文件"
    echo "  - 建议在项目根目录下运行此脚本"
    echo "  - 清理操作不可逆，请谨慎使用"
}

# 检查环境
check_environment() {
    if [ ! -f "package.json" ]; then
        log_error "请在项目根目录运行此脚本"
        exit 1
    fi
    
    if [ ! -d "cypress" ]; then
        log_error "未找到 cypress 目录，请确认这是一个 Cypress 项目"
        exit 1
    fi
}

# 清理历史报告
cleanup_reports() {
    local keep_count=$1
    
    log_info "开始清理历史测试报告，保留最近 $keep_count 次结果..."
    
    local total_cleaned=0
    
    # 清理 cypress/reports 目录中的历史报告
    if [ -d "$REPORTS_DIR" ]; then
        log_info "清理 $REPORTS_DIR 目录..."
        
        # 清理 HTML 报告文件（按时间戳排序）
        local html_files=($REPORTS_DIR/mochawesome_*.html)
        if [ ${#html_files[@]} -gt 0 ] && [ -f "${html_files[0]}" ]; then
            log_info "发现 ${#html_files[@]} 个 HTML 报告文件"
            
            # 按文件修改时间排序，删除旧文件
            local files_to_delete=$(printf '%s\n' "${html_files[@]}" | xargs ls -t | tail -n +$((keep_count + 1)))
            if [ -n "$files_to_delete" ]; then
                echo "$files_to_delete" | while read file; do
                    log_info "删除旧 HTML 报告: $(basename "$file")"
                    rm -f "$file"
                    ((total_cleaned++))
                done
            else
                log_info "HTML 报告文件数量未超过保留限制"
            fi
        else
            log_info "未找到 HTML 报告文件"
        fi
        
        # 清理 JSON 报告文件
        local json_files=($REPORTS_DIR/mochawesome_*.json)
        if [ ${#json_files[@]} -gt 0 ] && [ -f "${json_files[0]}" ]; then
            log_info "发现 ${#json_files[@]} 个 JSON 报告文件"
            
            local files_to_delete=$(printf '%s\n' "${json_files[@]}" | xargs ls -t | tail -n +$((keep_count + 1)))
            if [ -n "$files_to_delete" ]; then
                echo "$files_to_delete" | while read file; do
                    log_info "删除旧 JSON 报告: $(basename "$file")"
                    rm -f "$file"
                    ((total_cleaned++))
                done
            else
                log_info "JSON 报告文件数量未超过保留限制"
            fi
        else
            log_info "未找到 JSON 报告文件"
        fi
    else
        log_warning "$REPORTS_DIR 目录不存在"
    fi
    
    # 清理测试视频
    if [ -d "cypress/videos" ]; then
        log_info "清理测试视频..."
        
        local video_files=(cypress/videos/*.mp4)
        if [ ${#video_files[@]} -gt 0 ] && [ -f "${video_files[0]}" ]; then
            log_info "发现 ${#video_files[@]} 个视频文件"
            
            local files_to_delete=$(printf '%s\n' "${video_files[@]}" | xargs ls -t | tail -n +$((keep_count + 1)))
            if [ -n "$files_to_delete" ]; then
                echo "$files_to_delete" | while read file; do
                    log_info "删除旧视频: $(basename "$file")"
                    rm -f "$file"
                    ((total_cleaned++))
                done
            else
                log_info "视频文件数量未超过保留限制"
            fi
        else
            log_info "未找到视频文件"
        fi
    else
        log_info "未找到 cypress/videos 目录"
    fi
    
    # 清理截图目录（按目录修改时间）
    if [ -d "cypress/screenshots" ]; then
        log_info "清理失败截图..."
        
        # 获取所有截图子目录
        local screenshot_dirs=(cypress/screenshots/*/)
        if [ ${#screenshot_dirs[@]} -gt 0 ] && [ -d "${screenshot_dirs[0]}" ]; then
            log_info "发现 ${#screenshot_dirs[@]} 个截图目录"
            
            local dirs_to_delete=$(printf '%s\n' "${screenshot_dirs[@]}" | xargs ls -td | tail -n +$((keep_count + 1)))
            if [ -n "$dirs_to_delete" ]; then
                echo "$dirs_to_delete" | while read dir; do
                    log_info "删除旧截图目录: $(basename "$dir")"
                    rm -rf "$dir"
                    ((total_cleaned++))
                done
            else
                log_info "截图目录数量未超过保留限制"
            fi
        else
            log_info "未找到截图子目录"
        fi
    else
        log_info "未找到 cypress/screenshots 目录"
    fi
    
    # 可选：清理 public/reports 目录
    if [ -d "public/reports" ]; then
        log_info "清理合并报告..."
        local old_reports=$(find public/reports -name "*.html" -o -name "*.json" | wc -l)
        if [ "$old_reports" -gt 0 ]; then
            log_warning "发现 $old_reports 个合并报告文件"
            read -p "是否清理 public/reports 目录中的文件? (y/N): " -n 1 -r
            echo
            if [[ $REPLY =~ ^[Yy]$ ]]; then
                rm -f public/reports/*.html public/reports/*.json
                log_info "已清理 public/reports 目录"
                ((total_cleaned += old_reports))
            else
                log_info "跳过 public/reports 目录清理"
            fi
        else
            log_info "public/reports 目录为空"
        fi
    fi
    
    log_success "清理完成，共清理了约 $total_cleaned 个文件/目录"
}

# 显示当前状态
show_status() {
    log_info "当前测试文件状态:"
    
    echo
    echo "📊 报告文件:"
    if [ -d "$REPORTS_DIR" ]; then
        local html_count=$(find $REPORTS_DIR -name "mochawesome_*.html" 2>/dev/null | wc -l)
        local json_count=$(find $REPORTS_DIR -name "mochawesome_*.json" 2>/dev/null | wc -l)
        echo "  HTML 报告: $html_count 个"
        echo "  JSON 报告: $json_count 个"
    else
        echo "  报告目录不存在"
    fi
    
    echo
    echo "🎥 视频文件:"
    if [ -d "cypress/videos" ]; then
        local video_count=$(find cypress/videos -name "*.mp4" 2>/dev/null | wc -l)
        echo "  测试视频: $video_count 个"
    else
        echo "  视频目录不存在"
    fi
    
    echo
    echo "📸 截图文件:"
    if [ -d "cypress/screenshots" ]; then
        local screenshot_dirs=$(find cypress/screenshots -maxdepth 1 -type d 2>/dev/null | grep -v "^cypress/screenshots$" | wc -l)
        echo "  截图目录: $screenshot_dirs 个"
    else
        echo "  截图目录不存在"
    fi
    
    echo
}

# 主函数
main() {
    # 处理帮助和状态选项
    case ${1:-} in
        "help"|"-h"|"--help")
            show_usage
            exit 0
            ;;
        "status"|"-s"|"--status")
            check_environment
            show_status
            exit 0
            ;;
    esac
    
    local keep_count=${1:-$DEFAULT_KEEP_COUNT}
    
    # 参数验证
    if ! [[ "$keep_count" =~ ^[0-9]+$ ]] || [ "$keep_count" -lt 1 ]; then
        log_error "保留数量必须是大于0的整数"
        show_usage
        exit 1
    fi
    
    echo "========================================"
    echo "  DC-TOM 测试报告清理工具"
    echo "========================================"
    echo
    
    check_environment
    
    log_info "准备清理历史测试文件，保留最近 $keep_count 次测试结果"
    
    # 显示当前状态
    show_status
    
    # 确认操作
    read -p "确认继续清理? (y/N): " -n 1 -r
    echo
    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        log_info "操作已取消"
        exit 0
    fi
    
    # 执行清理
    cleanup_reports "$keep_count"
    
    echo
    log_success "报告清理完成!"
    echo
    echo "💡 提示:"
    echo "  - 运行 '$0 status' 查看当前文件状态"
    echo "  - 运行 '$0 help' 查看详细使用说明"
}

# 运行主函数
main "$@"