#!/bin/bash

# DC-TOM 快速测试脚本
# 简化版本，专门用于快速验证和调试

set -e

# 颜色输出
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
NC='\033[0m'

log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

# 快速环境检查
quick_check() {
    log_info "快速环境检查..."
    
    if [ ! -f "package.json" ]; then
        echo "错误: 请在项目根目录运行"
        exit 1
    fi
    
    log_success "环境检查通过"
}

# 快速构建
quick_build() {
    log_info "快速构建..."
    
    if [ ! -d "node_modules" ]; then
        log_info "安装依赖..."
        npm install
    fi
    
    if [ ! -d "dist" ]; then
        log_info "构建项目..."
        npm run build
    else
        log_warning "使用现有构建 (dist/ 已存在)"
    fi
    
    log_success "构建完成"
}

# 运行单个测试
run_single_test() {
    local spec_file=$1
    log_info "运行测试: $spec_file"
    
    # 启动服务器
    npm run preview -- --port 3000 &
    SERVER_PID=$!
    sleep 10
    
    # 运行测试
    npx cypress run \
        --browser chrome \
        --headless \
        --spec "$spec_file" \
        --reporter mochawesome \
        --reporter-options "reportDir=cypress/reports/quick,overwrite=false,html=true,json=true" \
        || log_warning "测试可能失败，但继续执行"
    
    # 停止服务器
    kill $SERVER_PID 2>/dev/null || true
    
    log_success "测试完成"
}

# 生成简单报告
quick_report() {
    log_info "生成快速报告..."
    
    mkdir -p public/quick-reports
    
    if [ -d "cypress/reports/quick" ]; then
        # 复制报告文件
        cp cypress/reports/quick/*.html public/quick-reports/ 2>/dev/null || true
        
        # 生成简单索引
        cat > public/quick-reports/index.html << 'EOF'
<!DOCTYPE html>
<html>
<head>
    <title>快速测试报告</title>
    <meta charset="utf-8">
</head>
<body>
    <h1>DC-TOM 快速测试报告</h1>
    <p>生成时间: <script>document.write(new Date().toLocaleString());</script></p>
    <ul>
EOF
        
        # 添加报告链接
        for file in public/quick-reports/*.html; do
            if [ "$file" != "public/quick-reports/index.html" ]; then
                filename=$(basename "$file")
                echo "        <li><a href=\"$filename\">$filename</a></li>" >> public/quick-reports/index.html
            fi
        done
        
        cat >> public/quick-reports/index.html << 'EOF'
    </ul>
</body>
</html>
EOF
        
        log_success "报告生成完成: public/quick-reports/"
        
        # 打开报告
        if command -v open &> /dev/null; then
            open public/quick-reports/index.html
        fi
    else
        log_warning "未找到测试报告"
    fi
}

# 主函数
main() {
    echo "================================"
    echo "  DC-TOM 快速测试脚本"
    echo "================================"
    
    local test_file=${1:-"cypress/e2e/spec.cy.js"}
    
    quick_check
    quick_build
    run_single_test "$test_file"
    quick_report
    
    echo ""
    log_success "快速测试完成!"
    echo "使用方法: $0 [测试文件路径]"
    echo "示例: $0 cypress/e2e/dashboard.cy.js"
}

# 清理函数
cleanup() {
    if [ ! -z "$SERVER_PID" ]; then
        kill $SERVER_PID 2>/dev/null || true
    fi
}

trap cleanup EXIT

main "$@"