#!/usr/bin/env node

/**
 * DC-TOM Cypress Studio 启动器
 * 跨平台支持 (Windows/macOS/Linux)
 * 功能: 先启动开发服务器，然后启动 Cypress Studio
 */

const { spawn, exec } = require('child_process');
const http = require('http');
const os = require('os');

// 颜色定义 (Windows CMD 支持的 ANSI 颜色)
const colors = {
    reset: '\x1b[0m',
    red: '\x1b[31m',
    green: '\x1b[32m',
    yellow: '\x1b[33m',
    blue: '\x1b[34m',
    cyan: '\x1b[36m'
};

// 日志函数
const log = {
    info: (msg) => console.log(`${colors.blue}ℹ️  ${msg}${colors.reset}`),
    success: (msg) => console.log(`${colors.green}✅ ${msg}${colors.reset}`),
    warning: (msg) => console.log(`${colors.yellow}⚠️  ${msg}${colors.reset}`),
    error: (msg) => console.log(`${colors.red}❌ ${msg}${colors.reset}`),
    plain: (msg) => console.log(msg)
};

// 全局变量
let devServerProcess = null;
let isWindows = os.platform() === 'win32';

// 检查端口是否被占用
function checkPort(port) {
    return new Promise((resolve) => {
        const server = http.createServer();
        server.listen(port, () => {
            server.close(() => resolve(false)); // 端口未被占用
        });
        server.on('error', () => resolve(true)); // 端口被占用
    });
}

// 等待开发服务器启动
function waitForServer(url, maxAttempts = 30) {
    return new Promise((resolve) => {
        let attempts = 0;
        
        const checkServer = () => {
            attempts++;
            
            const request = http.get(url, (res) => {
                if (res.statusCode === 200) {
                    resolve(true);
                } else if (attempts < maxAttempts) {
                    setTimeout(checkServer, 1000);
                } else {
                    resolve(false);
                }
            });
            
            request.on('error', () => {
                if (attempts < maxAttempts) {
                    process.stdout.write('.');
                    setTimeout(checkServer, 1000);
                } else {
                    resolve(false);
                }
            });
            
            request.setTimeout(2000, () => {
                request.destroy();
                if (attempts < maxAttempts) {
                    process.stdout.write('.');
                    setTimeout(checkServer, 1000);
                } else {
                    resolve(false);
                }
            });
        };
        
        checkServer();
    });
}

// 启动开发服务器
async function startDevServer() {
    const portOccupied = await checkPort(3000);
    
    if (portOccupied) {
        log.warning('端口 3000 已被占用，跳过开发服务器启动');
        log.info('如果需要重新启动，请先停止现有的服务');
        return true;
    }
    
    log.info('启动开发服务器...');
    
    return new Promise((resolve) => {
        // Windows 和 Unix 系统使用不同的命令
        const npmCommand = isWindows ? 'npm.cmd' : 'npm';
        
        devServerProcess = spawn(npmCommand, ['run', 'dev'], {
            cwd: process.cwd(),
            stdio: ['pipe', 'pipe', 'pipe'],
            shell: isWindows
        });
        
        devServerProcess.stdout.on('data', (data) => {
            const output = data.toString();
            if (output.includes('Local:') && output.includes('localhost:3000')) {
                log.success(`开发服务器启动成功 (PID: ${devServerProcess.pid})`);
                resolve(true);
            }
        });
        
        devServerProcess.stderr.on('data', (data) => {
            // Vite 的一些输出会到 stderr，但不一定是错误
            const output = data.toString();
            if (output.includes('Local:') && output.includes('localhost:3000')) {
                log.success(`开发服务器启动成功 (PID: ${devServerProcess.pid})`);
                resolve(true);
            }
        });
        
        devServerProcess.on('error', (error) => {
            log.error(`开发服务器启动失败: ${error.message}`);
            resolve(false);
        });
        
        devServerProcess.on('exit', (code) => {
            if (code !== 0) {
                log.error(`开发服务器异常退出，代码: ${code}`);
                resolve(false);
            }
        });
        
        // 设置超时
        setTimeout(() => {
            log.info('等待开发服务器启动...');
            waitForServer('http://localhost:3000').then((serverReady) => {
                if (serverReady) {
                    log.success('开发服务器已就绪');
                    resolve(true);
                } else {
                    log.error('开发服务器启动超时');
                    resolve(false);
                }
            });
        }, 3000);
    });
}

// 启动 Cypress Studio
function startCypressStudio() {
    log.info('启动 Cypress Studio...');
    log.info('Cypress Studio 将在新窗口中打开');
    log.info('您可以在 Studio 中录制和编辑测试用例');
    
    const npxCommand = isWindows ? 'npx.cmd' : 'npx';
    
    const cypressProcess = spawn(npxCommand, ['cypress', 'open', '--config', 'experimentalStudio=true'], {
        cwd: process.cwd(),
        stdio: 'inherit',
        shell: isWindows
    });
    
    cypressProcess.on('error', (error) => {
        log.error(`Cypress Studio 启动失败: ${error.message}`);
        cleanup();
        process.exit(1);
    });
    
    cypressProcess.on('exit', (code) => {
        log.info('Cypress Studio 已关闭');
        cleanup();
        process.exit(code || 0);
    });
}

// 清理函数
function cleanup() {
    log.info('正在清理进程...');
    
    if (devServerProcess && !devServerProcess.killed) {
        log.info(`停止开发服务器 (PID: ${devServerProcess.pid})`);
        
        if (isWindows) {
            // Windows 需要强制终止整个进程树
            exec(`taskkill /pid ${devServerProcess.pid} /T /F`, (error) => {
                if (error) {
                    log.warning('无法自动停止开发服务器，请手动终止');
                }
            });
        } else {
            devServerProcess.kill('SIGTERM');
        }
    }
}

// 显示使用说明
function showUsage() {
    log.plain('用法: node start-studio.js [选项]');
    log.plain('');
    log.plain('选项:');
    log.plain('  --help, -h    显示此帮助信息');
    log.plain('');
    log.plain('功能:');
    log.plain('  1. 自动启动开发服务器 (localhost:3000)');
    log.plain('  2. 等待服务器就绪');
    log.plain('  3. 启动 Cypress Studio');
    log.plain('');
    log.plain('注意:');
    log.plain('  - 支持 Windows/macOS/Linux');
    log.plain('  - 如果端口 3000 已被占用，将跳过开发服务器启动');
    log.plain('  - 关闭 Cypress Studio 时，开发服务器也会自动停止');
    log.plain('  - 使用 Ctrl+C 可以同时停止两个服务');
}

// 主函数
async function main() {
    // 处理命令行参数
    if (process.argv.includes('--help') || process.argv.includes('-h')) {
        showUsage();
        return;
    }
    
    console.log();
    log.info('🚀 DC-TOM Cypress Studio 启动器');
    log.info(`📦 运行平台: ${os.platform()} ${os.arch()}`);
    console.log();
    
    // 启动开发服务器
    const serverStarted = await startDevServer();
    
    if (serverStarted) {
        console.log();
        log.success('开发环境准备就绪');
        console.log();
        
        // 启动 Cypress Studio
        startCypressStudio();
    } else {
        log.error('无法启动开发服务器，Cypress Studio 启动已取消');
        process.exit(1);
    }
}

// 设置退出处理
process.on('SIGINT', () => {
    log.info('\n收到中断信号，正在退出...');
    cleanup();
    process.exit(0);
});

process.on('SIGTERM', () => {
    log.info('\n收到终止信号，正在退出...');
    cleanup();
    process.exit(0);
});

process.on('exit', () => {
    cleanup();
});

// 运行主函数
main().catch((error) => {
    log.error(`启动器发生错误: ${error.message}`);
    cleanup();
    process.exit(1);
});