#!/bin/bash

# DC-TOM Cypress Studio 启动脚本
# 功能: 先启动开发服务器，然后启动 Cypress Studio

set -e  # 遇到错误时退出

# 颜色定义
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 日志函数
log_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

log_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

log_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

log_error() {
    echo -e "${RED}❌ $1${NC}"
}

# 检查端口是否被占用
check_port() {
    local port=$1
    if lsof -Pi :$port -sTCP:LISTEN -t >/dev/null 2>&1; then
        return 0  # 端口被占用
    else
        return 1  # 端口未被占用
    fi
}

# 等待开发服务器启动
wait_for_dev_server() {
    local max_attempts=30
    local attempt=1
    
    log_info "等待开发服务器启动 (localhost:3000)..."
    
    while [ $attempt -le $max_attempts ]; do
        if curl -s http://localhost:3000 > /dev/null 2>&1; then
            log_success "开发服务器已启动并运行在 http://localhost:3000"
            return 0
        fi
        
        echo -n "."
        sleep 1
        attempt=$((attempt + 1))
    done
    
    log_error "开发服务器启动超时"
    return 1
}

# 启动开发服务器
start_dev_server() {
    if check_port 3000; then
        log_warning "端口 3000 已被占用，跳过开发服务器启动"
        log_info "如果需要重新启动，请先停止现有的服务"
        return 0
    fi
    
    log_info "启动开发服务器..."
    npm run dev &
    DEV_PID=$!
    
    # 等待服务器启动
    if wait_for_dev_server; then
        log_success "开发服务器启动成功 (PID: $DEV_PID)"
        return 0
    else
        log_error "开发服务器启动失败"
        if [ ! -z "$DEV_PID" ]; then
            kill $DEV_PID 2>/dev/null || true
        fi
        return 1
    fi
}

# 启动 Cypress Studio
start_cypress_studio() {
    log_info "启动 Cypress Studio..."
    log_info "Cypress Studio 将在新窗口中打开"
    log_info "您可以在 Studio 中录制和编辑测试用例"
    
    # 启动 Cypress Studio
    npx cypress open --config experimentalStudio=true
}

# 清理函数
cleanup() {
    log_info "正在清理进程..."
    if [ ! -z "$DEV_PID" ] && kill -0 $DEV_PID 2>/dev/null; then
        log_info "停止开发服务器 (PID: $DEV_PID)"
        kill $DEV_PID 2>/dev/null || true
    fi
}

# 设置退出处理
trap cleanup EXIT INT TERM

# 主函数
main() {
    echo
    log_info "🚀 DC-TOM Cypress Studio 启动器"
    echo
    
    # 检查依赖
    if ! command -v npm >/dev/null 2>&1; then
        log_error "npm 未安装或不在 PATH 中"
        exit 1
    fi
    
    if ! command -v npx >/dev/null 2>&1; then
        log_error "npx 未安装或不在 PATH 中"
        exit 1
    fi
    
    # 启动开发服务器
    if start_dev_server; then
        echo
        log_success "开发环境准备就绪"
        echo
        
        # 启动 Cypress Studio
        start_cypress_studio
    else
        log_error "无法启动开发服务器，Cypress Studio 启动已取消"
        exit 1
    fi
}

# 显示使用说明
show_usage() {
    echo "用法: $0"
    echo ""
    echo "功能:"
    echo "  1. 自动启动开发服务器 (localhost:3000)"
    echo "  2. 等待服务器就绪"
    echo "  3. 启动 Cypress Studio"
    echo ""
    echo "注意:"
    echo "  - 如果端口 3000 已被占用，将跳过开发服务器启动"
    echo "  - 关闭 Cypress Studio 时，开发服务器也会自动停止"
    echo "  - 使用 Ctrl+C 可以同时停止两个服务"
}

# 处理命令行参数
if [ "$1" = "--help" ] || [ "$1" = "-h" ]; then
    show_usage
    exit 0
fi

# 运行主函数
main "$@"