/**
 * 测试模板引擎
 * 负责管理和渲染测试模板
 */

const fs = require('fs');
const path = require('path');

class TestTemplateEngine {
  constructor() {
    this.templates = new Map();
    this.loadTemplates();
  }

  /**
   * 加载所有测试模板
   */
  loadTemplates() {
    const templatesDir = path.join(__dirname, 'templates');
    
    // 基础模板
    this.templates.set('base', this.createBaseTemplate());
    this.templates.set('beforeEach', this.createBeforeEachTemplate());
    this.templates.set('uiValidation', this.createUIValidationTemplate());
    this.templates.set('interaction', this.createInteractionTemplate());
    this.templates.set('dataValidation', this.createDataValidationTemplate());
    this.templates.set('errorHandling', this.createErrorHandlingTemplate());
    this.templates.set('responsive', this.createResponsiveTemplate());
    this.templates.set('performance', this.createPerformanceTemplate());
  }

  /**
   * 基础测试模板
   */
  createBaseTemplate() {
    return `/// <reference types="cypress" />

describe('{{testSuiteName}}', () => {
  {{beforeEachBlock}}

  {{testScenarios}}
})`;
  }

  /**
   * beforeEach模板
   */
  createBeforeEachTemplate() {
    return `beforeEach(() => {
    {{#if login}}
    // 模拟登录状态
    cy.mockLogin()
    {{/if}}
    
    {{#if visit}}
    // 访问页面
    cy.visit('/#/{{visit}}')
    {{/if}}
    
    {{#each waitFor}}
    // 等待页面加载
    cy.get('{{this}}').should('be.visible')
    {{/each}}
  })`;
  }

  /**
   * UI验证模板
   */
  createUIValidationTemplate() {
    return `it('{{scenarioName}}', () => {
    {{#each steps}}
    {{#if (eq action "verify")}}
    // {{description}}
    cy.get('{{target}}').should('{{assertion.assertion}}'{{#if assertion.value}}, '{{assertion.value}}'{{/if}})
    {{/if}}
    {{/each}}
  })`;
  }

  /**
   * 交互测试模板
   */
  createInteractionTemplate() {
    return `it('{{scenarioName}}', () => {
    {{#each steps}}
    {{#if (eq action "click")}}
    // {{description}}
    cy.get('{{target}}').click()
    {{/if}}
    {{#if (eq action "type")}}
    // {{description}}
    cy.get('{{target}}'){{#if options.clear}}.clear(){{/if}}.type('{{value}}')
    {{/if}}
    {{#if (eq action "select")}}
    // {{description}}
    cy.get('{{target}}').click()
    cy.get('.el-select-dropdown__item').contains('{{value}}').click()
    {{/if}}
    {{#if (eq action "wait")}}
    // {{description}}
    cy.wait({{value}})
    {{/if}}
    {{/each}}
    
    {{#each expectedResults}}
    // 验证结果
    cy.get('{{target}}').should('{{assertion}}'{{#if value}}, '{{value}}'{{/if}})
    {{/each}}
  })`;
  }

  /**
   * 数据验证模板
   */
  createDataValidationTemplate() {
    return `it('{{scenarioName}}', () => {
    {{#each steps}}
    {{#if (eq action "intercept")}}
    // 模拟API响应
    cy.intercept('{{value.method}}', '{{value.url}}', {{#if value.response}}{{json value.response}}{{else}}{ fixture: '{{value.fixture}}' }{{/if}}).as('{{value.alias}}')
    {{/if}}
    {{/each}}
    
    {{#each steps}}
    {{#unless (eq action "intercept")}}
    {{> (concat action "Step") this}}
    {{/unless}}
    {{/each}}
    
    {{#each expectedResults}}
    cy.get('{{target}}').should('{{assertion}}'{{#if value}}, '{{value}}'{{/if}})
    {{/each}}
  })`;
  }

  /**
   * 错误处理模板
   */
  createErrorHandlingTemplate() {
    return `it('{{scenarioName}}', () => {
    {{#each steps}}
    {{#if (eq action "intercept")}}
    // 模拟错误响应
    cy.intercept('{{value.method}}', '{{value.url}}', {
      statusCode: {{value.statusCode}},
      body: { error: '{{value.error}}' }
    }).as('{{value.alias}}')
    {{/if}}
    {{/each}}
    
    {{#each steps}}
    {{#unless (eq action "intercept")}}
    {{> (concat action "Step") this}}
    {{/unless}}
    {{/each}}
    
    {{#each expectedResults}}
    // 验证错误处理
    cy.get('{{target}}').should('{{assertion}}'{{#if value}}, '{{value}}'{{/if}})
    {{/each}}
  })`;
  }

  /**
   * 响应式设计模板
   */
  createResponsiveTemplate() {
    return `it('{{scenarioName}}', () => {
    const viewports = [
      { width: 1920, height: 1080 }, // 桌面
      { width: 1280, height: 720 },  // 笔记本
      { width: 768, height: 1024 },  // 平板
      { width: 375, height: 667 }    // 手机
    ]
    
    viewports.forEach(viewport => {
      cy.viewport(viewport.width, viewport.height)
      
      {{#each expectedResults}}
      cy.get('{{target}}').should('{{assertion}}'{{#if value}}, '{{value}}'{{/if}})
      {{/each}}
    })
  })`;
  }

  /**
   * 性能测试模板
   */
  createPerformanceTemplate() {
    return `it('{{scenarioName}}', () => {
    // 记录页面加载时间
    cy.window().then((win) => {
      const performance = win.performance
      const navigation = performance.getEntriesByType('navigation')[0]
      
      // 验证页面加载时间
      expect(navigation.loadEventEnd - navigation.loadEventStart).to.be.lessThan({{value}})
    })
    
    {{#each expectedResults}}
    cy.get('{{target}}').should('{{assertion}}'{{#if value}}, '{{value}}'{{/if}})
    {{/each}}
  })`;
  }

  /**
   * 渲染模板
   * @param {string} templateName - 模板名称
   * @param {object} context - 模板上下文数据
   * @returns {string} 渲染后的代码
   */
  render(templateName, context) {
    const template = this.templates.get(templateName);
    if (!template) {
      throw new Error(`Template '${templateName}' not found`);
    }

    return this.processTemplate(template, context);
  }

  /**
   * 处理模板字符串
   * @param {string} template - 模板字符串
   * @param {object} context - 上下文数据
   * @returns {string} 处理后的字符串
   */
  processTemplate(template, context) {
    // 简单的模板处理逻辑（可以替换为更强大的模板引擎如Handlebars）
    let result = template;

    // 处理基本变量替换 {{variable}}
    result = result.replace(/\{\{([^}]+)\}\}/g, (match, variable) => {
      const value = this.getNestedValue(context, variable.trim());
      return value !== undefined ? value : match;
    });

    // 处理条件语句 {{#if condition}}
    result = this.processConditionals(result, context);

    // 处理循环语句 {{#each array}}
    result = this.processLoops(result, context);

    return result;
  }

  /**
   * 获取嵌套对象的值
   */
  getNestedValue(obj, path) {
    return path.split('.').reduce((current, key) => {
      return current && current[key] !== undefined ? current[key] : undefined;
    }, obj);
  }

  /**
   * 处理条件语句
   */
  processConditionals(template, context) {
    return template.replace(/\{\{#if\s+([^}]+)\}\}([\s\S]*?)\{\{\/if\}\}/g, (match, condition, content) => {
      const value = this.getNestedValue(context, condition.trim());
      return value ? content : '';
    });
  }

  /**
   * 处理循环语句
   */
  processLoops(template, context) {
    return template.replace(/\{\{#each\s+([^}]+)\}\}([\s\S]*?)\{\{\/each\}\}/g, (match, arrayPath, content) => {
      const array = this.getNestedValue(context, arrayPath.trim());
      if (!Array.isArray(array)) return '';

      return array.map((item, index) => {
        const itemContext = { ...context, this: item, '@index': index };
        return this.processTemplate(content, itemContext);
      }).join('\n');
    });
  }

  /**
   * 获取所有可用模板
   */
  getAvailableTemplates() {
    return Array.from(this.templates.keys());
  }
}

module.exports = TestTemplateEngine;