#!/bin/bash

# DC-TOM 测试生成器集成脚本
# 用于自动化测试生成和执行流程

set -e

# 颜色输出
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# 项目配置
PROJECT_ROOT="$(pwd)"
GENERATOR_DIR="$PROJECT_ROOT/cypress/test-generator"
GENERATED_DIR="$PROJECT_ROOT/cypress/e2e/generated"
INPUT_DIR="$PROJECT_ROOT/test-descriptions"

# 日志函数
log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 显示帮助信息
show_help() {
    echo "DC-TOM 测试生成器集成脚本"
    echo ""
    echo "用法: $0 <command> [options]"
    echo ""
    echo "命令:"
    echo "  setup                    初始化测试生成器环境"
    echo "  example                  生成示例JSON描述文件"
    echo "  nl-example              生成自然语言描述示例文件"
    echo "  generate <file>          从JSON生成测试代码"
    echo "  parse <file>             从自然语言描述生成测试代码"
    echo "  generate-all             批量生成所有JSON描述文件"
    echo "  parse-all               批量解析所有自然语言描述文件"
    echo "  validate <file>          验证JSON描述文件"
    echo "  validate-all             验证所有JSON描述文件"
    echo "  run-generated            运行生成的测试"
    echo "  clean                    清理生成的测试文件"
    echo "  web                      启动Web界面"
    echo "  help                     显示帮助信息"
    echo ""
    echo "选项:"
    echo "  -o, --output <file>      指定输出文件"
    echo "  -f, --force              强制覆盖已存在的文件"
    echo ""
    echo "示例:"
    echo "  $0 setup"
    echo "  $0 example"
    echo "  $0 nl-example"
    echo "  $0 generate test-descriptions/collector-test.json"
    echo "  $0 parse test-descriptions/natural-language-example.txt"
    echo "  $0 generate-all"
    echo "  $0 parse-all"
    echo "  $0 run-generated"
}

# 初始化环境
setup_environment() {
    log_info "初始化测试生成器环境..."
    
    # 创建必要的目录
    mkdir -p "$INPUT_DIR"
    mkdir -p "$GENERATED_DIR"
    
    # 检查Node.js环境
    if ! command -v node &> /dev/null; then
        log_error "Node.js 未安装，请先安装 Node.js 18+"
        exit 1
    fi
    
    log_info "Node.js 版本: $(node -v)"
    
    # 设置CLI工具权限
    chmod +x "$GENERATOR_DIR/cli.js"
    
    log_success "环境初始化完成"
    log_info "测试描述目录: $INPUT_DIR"
    log_info "生成的测试目录: $GENERATED_DIR"
    log_info "生成器目录: $GENERATOR_DIR"
}

# 生成示例文件
generate_examples() {
    log_info "生成示例JSON描述文件..."
    
    # 确保输入目录存在
    mkdir -p "$INPUT_DIR"
    
    # 使用CLI工具生成示例
    cd "$GENERATOR_DIR"
    node cli.js example
    
    # 移动示例文件到输入目录
    if [ -f "test-description-example.json" ]; then
        mv "test-description-example.json" "$INPUT_DIR/"
        log_success "示例文件已生成: $INPUT_DIR/test-description-example.json"
    fi
    
    # 生成多个模块的示例
    create_module_examples
    
    cd "$PROJECT_ROOT"
}

# 创建各模块的示例文件
create_module_examples() {
    log_info "创建各模块示例文件..."
    
    # 仪表盘示例
    cat > "$INPUT_DIR/dashboard-test.json" << 'EOF'
{
  "testSuite": {
    "name": "仪表盘功能测试",
    "module": "仪表盘",
    "description": "测试仪表盘页面的核心功能",
    "scenarios": [
      {
        "name": "应该显示健康度指标",
        "type": "ui",
        "priority": "high",
        "steps": [
          {
            "action": "verify",
            "target": "[data-testid=\"dashboard-health-score\"]",
            "value": { "assertion": "be.visible" },
            "description": "检查健康度评分"
          },
          {
            "action": "verify",
            "target": "[data-testid=\"dashboard-chart-container\"]",
            "value": { "assertion": "be.visible" },
            "description": "检查图表容器"
          }
        ]
      }
    ]
  }
}
EOF

    # 设备管理示例
    cat > "$INPUT_DIR/device-management-test.json" << 'EOF'
{
  "testSuite": {
    "name": "设备管理功能测试",
    "module": "设备管理",
    "description": "测试设备管理页面的搜索和管理功能",
    "scenarios": [
      {
        "name": "应该能够搜索设备",
        "type": "interaction",
        "priority": "high",
        "steps": [
          {
            "action": "type",
            "target": "[data-testid=\"equipment-device-name-input\"]",
            "value": "测试设备",
            "options": { "clear": true },
            "description": "输入设备名称"
          },
          {
            "action": "click",
            "target": "{searchButton}",
            "description": "点击搜索按钮"
          },
          {
            "action": "wait",
            "value": 1000,
            "description": "等待搜索结果"
          }
        ],
        "expectedResults": [
          {
            "target": "[data-testid=\"equipment-management-table\"]",
            "assertion": "be.visible"
          }
        ]
      }
    ]
  }
}
EOF

    log_success "模块示例文件创建完成"
}

# 验证JSON文件
validate_json() {
    local file_path="$1"
    
    if [ ! -f "$file_path" ]; then
        log_error "文件不存在: $file_path"
        return 1
    fi
    
    log_info "验证文件: $file_path"
    
    cd "$GENERATOR_DIR"
    if node cli.js validate "$file_path"; then
        log_success "验证通过: $(basename "$file_path")"
        return 0
    else
        log_error "验证失败: $(basename "$file_path")"
        return 1
    fi
}

# 验证所有JSON文件
validate_all() {
    log_info "验证所有JSON描述文件..."
    
    local success_count=0
    local fail_count=0
    
    if [ ! -d "$INPUT_DIR" ]; then
        log_warning "输入目录不存在: $INPUT_DIR"
        return 1
    fi
    
    for json_file in "$INPUT_DIR"/*.json; do
        if [ -f "$json_file" ]; then
            if validate_json "$json_file"; then
                ((success_count++))
            else
                ((fail_count++))
            fi
        fi
    done
    
    log_info "验证完成: 成功 $success_count 个，失败 $fail_count 个"
    
    if [ $fail_count -eq 0 ]; then
        log_success "所有文件验证通过"
        return 0
    else
        log_error "有 $fail_count 个文件验证失败"
        return 1
    fi
}

# 生成测试代码
generate_test() {
    local input_file="$1"
    local output_file="$2"
    
    if [ ! -f "$input_file" ]; then
        log_error "输入文件不存在: $input_file"
        return 1
    fi
    
    log_info "生成测试代码: $(basename "$input_file")"
    
    cd "$GENERATOR_DIR"
    
    if [ -n "$output_file" ]; then
        node cli.js generate "$input_file" -o "$output_file"
    else
        node cli.js generate "$input_file"
    fi
    
    if [ $? -eq 0 ]; then
        log_success "测试代码生成成功"
        return 0
    else
        log_error "测试代码生成失败"
        return 1
    fi
}

# 批量生成所有测试
generate_all() {
    log_info "批量生成所有测试代码..."
    
    # 确保生成目录存在
    mkdir -p "$GENERATED_DIR"
    
    local success_count=0
    local fail_count=0
    
    if [ ! -d "$INPUT_DIR" ]; then
        log_warning "输入目录不存在: $INPUT_DIR，先生成示例文件"
        generate_examples
    fi
    
    for json_file in "$INPUT_DIR"/*.json; do
        if [ -f "$json_file" ]; then
            local base_name=$(basename "$json_file" .json)
            local output_file="$GENERATED_DIR/${base_name}.cy.js"
            
            if generate_test "$json_file" "$output_file"; then
                ((success_count++))
            else
                ((fail_count++))
            fi
        fi
    done
    
    log_info "生成完成: 成功 $success_count 个，失败 $fail_count 个"
    
    if [ $success_count -gt 0 ]; then
        log_success "生成的测试文件位于: $GENERATED_DIR"
        ls -la "$GENERATED_DIR"
    fi
}

# 运行生成的测试
run_generated_tests() {
    log_info "运行生成的测试..."
    
    if [ ! -d "$GENERATED_DIR" ] || [ -z "$(ls -A "$GENERATED_DIR" 2>/dev/null)" ]; then
        log_warning "没有找到生成的测试文件，先生成测试"
        generate_all
    fi
    
    # 统计生成的测试文件
    local test_count=$(find "$GENERATED_DIR" -name "*.cy.js" | wc -l)
    log_info "找到 $test_count 个生成的测试文件"
    
    if [ $test_count -eq 0 ]; then
        log_error "没有可运行的测试文件"
        return 1
    fi
    
    # 运行Cypress测试
    log_info "使用Cypress运行生成的测试..."
    
    npx cypress run \
        --browser chrome \
        --headless \
        --spec "cypress/e2e/generated/*.cy.js" \
        --config video=true,screenshotOnRunFailure=true \
        --reporter mochawesome \
        --reporter-options "reportDir=cypress/reports/generated,overwrite=false,html=true,json=true"
    
    if [ $? -eq 0 ]; then
        log_success "生成的测试运行完成"
    else
        log_warning "部分测试可能失败，请查看报告"
    fi
}

# 清理生成的文件
clean_generated() {
    log_info "清理生成的测试文件..."
    
    if [ -d "$GENERATED_DIR" ]; then
        rm -rf "$GENERATED_DIR"/*
        log_success "生成的测试文件已清理"
    else
        log_info "没有需要清理的文件"
    fi
}

# 启动Web界面
start_web_interface() {
    log_info "启动Web界面..."
    
    local web_file="$GENERATOR_DIR/web-interface.html"
    
    if [ ! -f "$web_file" ]; then
        log_error "Web界面文件不存在: $web_file"
        return 1
    fi
    
    log_success "正在打开Web界面..."
    
    # 根据操作系统选择打开方式
    if command -v open &> /dev/null; then
        # macOS
        open "$web_file"
    elif command -v xdg-open &> /dev/null; then
        # Linux
        xdg-open "$web_file"
    elif command -v start &> /dev/null; then
        # Windows
        start "$web_file"
    else
        log_info "请手动打开: $web_file"
    fi
}

# 主函数
main() {
    case "${1:-help}" in
        setup)
            setup_environment
            ;;
        example)
            generate_examples
            ;;
        generate)
            if [ -z "$2" ]; then
                log_error "请指定输入文件"
                show_help
                exit 1
            fi
            generate_test "$2" "$3"
            ;;
        generate-all)
            generate_all
            ;;
        validate)
            if [ -z "$2" ]; then
                log_error "请指定要验证的文件"
                show_help
                exit 1
            fi
            validate_json "$2"
            ;;
        validate-all)
            validate_all
            ;;
        run-generated)
            run_generated_tests
            ;;
        clean)
            clean_generated
            ;;
        web)
            start_web_interface
            ;;
        help)
            show_help
            ;;
        *)
            log_error "未知命令: $1"
            show_help
            exit 1
            ;;
    esac
}

# 运行主函数
main "$@"