/// <reference types="cypress" />

describe('布袋周期管理功能测试', () => {
  beforeEach(() => {
    // 模拟登录状态
    cy.mockLogin()
    
    // 访问布袋周期页面
    cy.visit('/#/collectorList')
    
    // 等待页面加载完成
    cy.get('[data-testid="collector-list-container"]').should('be.visible')
  })

  it('应该显示布袋周期页面的所有核心组件', () => {
    // 检查主容器
    cy.get('[data-testid="collector-list-container"]').should('be.visible')
    
    // 检查内容区域
    cy.get('[data-testid="collector-list-content"]').should('be.visible')
    
    // 检查搜索表单
    cy.get('[data-testid="collector-list-search-form"]').should('be.visible')
    
    // 检查表格容器
    cy.get('[data-testid="collector-table-container"]').should('be.visible')
    
    // 检查通用表格组件
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
  })

  it('应该显示搜索表单的所有输入字段', () => {
    // 检查仓室输入框
    cy.get('[data-testid="collector-compart-input"]')
      .should('be.visible')
      .and('have.attr', 'placeholder', '请输入仓室名称')
    
    // 检查除尘器名称输入框
    cy.get('[data-testid="collector-duster-name-input"]')
      .should('be.visible')
      .and('have.attr', 'placeholder', '请输入除尘器名称')
    
    // 检查日期选择器
    cy.get('[data-testid="collector-date-picker"]').should('be.visible')
    
    // 检查按钮
    cy.get('[data-testid="collector-reset-button"]').should('be.visible')
    cy.get('[data-testid="collector-search-button"]').should('be.visible')
    cy.get('[data-testid="collector-analysis-button"]').should('be.visible')
  })

  it('应该能够进行搜索操作', () => {
    // 输入仓室名称
    cy.get('[data-testid="collector-compart-input"]')
      .clear()
      .type('测试仓室')
    
    // 输入除尘器名称
    cy.get('[data-testid="collector-duster-name-input"]')
      .clear()
      .type('测试除尘器')
    
    // 点击查询按钮
    cy.get('[data-testid="collector-search-button"]').click()
    
    // 等待搜索结果加载
    cy.wait(1000)
    
    // 验证表格仍然可见
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
  })

  it('应该能够重置搜索条件', () => {
    // 先输入一些搜索条件
    cy.get('[data-testid="collector-compart-input"]').type('测试数据')
    cy.get('[data-testid="collector-duster-name-input"]').type('测试数据')
    
    // 点击重置按钮
    cy.get('[data-testid="collector-reset-button"]').click()
    
    // 验证输入框已清空
    cy.get('[data-testid="collector-compart-input"]').should('have.value', '')
    cy.get('[data-testid="collector-duster-name-input"]').should('have.value', '')
    
    // 验证表格仍然可见
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
  })

  it('应该能够设置日期范围', () => {
    // 点击日期选择器
    cy.get('[data-testid="collector-date-picker"]').click()
    
    // 等待日期选择器弹出
    cy.get('.el-picker-panel').should('be.visible')
    
    // 选择开始日期（这里需要根据实际的日期选择器实现调整）
    cy.get('.el-picker-panel').within(() => {
      // 选择今天的日期作为开始日期
      cy.get('.el-date-table td.available').first().click()
      
      // 选择明天的日期作为结束日期
      cy.get('.el-date-table td.available').eq(1).click()
    })
    
    // 验证日期选择器关闭
    cy.get('.el-picker-panel').should('not.exist')
  })

  it('应该显示表格数据', () => {
    // 等待表格加载
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
    
    // 检查表格是否有数据行
    cy.get('[data-testid="collector-common-table"]').within(() => {
      // 检查表格行是否存在（即使为空数据）
      cy.get('.el-table__body-wrapper').should('be.visible')
    })
  })

  it('应该能够点击除尘器名称打开分析对话框', () => {
    // 等待表格加载
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
    
    // 查找除尘器名称链接并双击
    cy.get('[data-testid="collector-duster-name-link"]').first().dblclick()
    
    // 验证对话框打开
    cy.get('.dustListDialog').should('be.visible')
    cy.get('.el-dialog__title').should('contain', '更换周期分析')
    
    // 验证对话框内容
    cy.get('.dustListDialog').within(() => {
      // 检查除尘器名称选择器
      cy.get('.input-group').should('be.visible')
      cy.get('.el-select').should('be.visible')
      
      // 检查图表区域
      cy.get('.echartBox').should('be.visible')
    })
  })

  it('应该能够通过分析按钮打开对话框', () => {
    // 点击更换周期分析按钮
    cy.get('[data-testid="collector-analysis-button"]').click()
    
    // 验证对话框打开
    cy.get('.dustListDialog').should('be.visible')
    cy.get('.el-dialog__title').should('contain', '更换周期分析')
  })

  it('应该能够关闭分析对话框', () => {
    // 打开对话框
    cy.get('[data-testid="collector-analysis-button"]').click()
    cy.get('.dustListDialog').should('be.visible')
    
    // 点击关闭按钮
    cy.get('.dustListDialog .el-dialog__headerbtn').click()
    
    // 验证对话框关闭
    cy.get('.dustListDialog').should('not.exist')
  })

  it('应该能够切换除尘器选择', () => {
    // 打开对话框
    cy.get('[data-testid="collector-analysis-button"]').click()
    cy.get('.dustListDialog').should('be.visible')
    
    // 点击除尘器选择器
    cy.get('.dustListDialog .el-select').click()
    
    // 等待下拉选项出现
    cy.get('.el-select-dropdown').should('be.visible')
    
    // 选择第一个选项（如果存在）
    cy.get('.el-select-dropdown .el-select-dropdown__item').first().click()
    
    // 验证选择器关闭
    cy.get('.el-select-dropdown').should('not.exist')
  })

  it('应该验证表格分页功能', () => {
    // 等待表格加载
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
    
    // 检查分页组件是否存在
    cy.get('[data-testid="collector-common-table"]').within(() => {
      // 查找分页组件
      cy.get('.el-pagination').should('be.visible')
    })
  })

  it('应该验证表格列标题', () => {
    // 等待表格加载
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
    
    // 检查表格列标题
    cy.get('[data-testid="collector-common-table"]').within(() => {
      // 检查主要列标题
      cy.get('.el-table__header').should('contain', '序号')
      cy.get('.el-table__header').should('contain', '除尘器名称')
      cy.get('.el-table__header').should('contain', '仓室')
      cy.get('.el-table__header').should('contain', '布袋位置（排/列）')
      cy.get('.el-table__header').should('contain', '布袋更换提醒时间')
      cy.get('.el-table__header').should('contain', '更换时间')
      cy.get('.el-table__header').should('contain', '更换人')
      cy.get('.el-table__header').should('contain', '更换周期(与上次更换比)')
    })
  })

  it('应该处理空数据状态', () => {
    // 输入一个不存在的搜索条件
    cy.get('[data-testid="collector-compart-input"]').type('不存在的仓室')
    cy.get('[data-testid="collector-search-button"]').click()
    
    // 等待搜索结果
    cy.wait(1000)
    
    // 验证表格仍然可见（即使没有数据）
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
  })

  it('应该验证页面响应式设计', () => {
    // 测试不同屏幕尺寸
    const viewports = [
      { width: 1920, height: 1080 }, // 桌面
      { width: 1280, height: 720 },  // 笔记本
      { width: 768, height: 1024 },  // 平板
      { width: 375, height: 667 }    // 手机
    ]
    
    viewports.forEach(viewport => {
      cy.viewport(viewport.width, viewport.height)
      
      // 验证主要组件仍然可见
      cy.get('[data-testid="collector-list-container"]').should('be.visible')
      cy.get('[data-testid="collector-list-search-form"]').should('be.visible')
      cy.get('[data-testid="collector-table-container"]').should('be.visible')
    })
  })

  it('应该验证搜索表单的输入验证', () => {
    // 测试输入框的清除功能
    cy.get('[data-testid="collector-compart-input"]').type('测试输入')
    cy.get('[data-testid="collector-compart-input"]').clear()
    cy.get('[data-testid="collector-compart-input"]').should('have.value', '')
    
    cy.get('[data-testid="collector-duster-name-input"]').type('测试输入')
    cy.get('[data-testid="collector-duster-name-input"]').clear()
    cy.get('[data-testid="collector-duster-name-input"]').should('have.value', '')
  })

  it('应该验证表格数据的交互性', () => {
    // 等待表格加载
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
    
    // 检查除尘器名称链接的样式
    cy.get('[data-testid="collector-duster-name-link"]').first().should('have.class', 'health-score')
    cy.get('[data-testid="collector-duster-name-link"]').first().should('have.class', 'green-color')
  })

  it('应该验证页面加载性能', () => {
    // 记录页面加载时间
    cy.window().then((win) => {
      const performance = win.performance
      const navigation = performance.getEntriesByType('navigation')[0]
      
      // 验证页面加载时间在合理范围内（小于10秒）
      expect(navigation.loadEventEnd - navigation.loadEventStart).to.be.lessThan(10000)
    })
  })

  it('应该验证错误处理', () => {
    // 模拟网络错误（通过拦截API请求）
    cy.intercept('GET', '**/bag/cycle/getReplaceListPage', {
      statusCode: 500,
      body: { error: '服务器错误' }
    }).as('getCollectorListError')
    
    // 触发搜索操作
    cy.get('[data-testid="collector-search-button"]').click()
    
    // 等待错误响应
    cy.wait('@getCollectorListError')
    
    // 验证页面仍然可用
    cy.get('[data-testid="collector-list-container"]').should('be.visible')
  })

  it('应该验证数据刷新功能', () => {
    // 记录初始数据
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
    
    // 等待一段时间后验证页面仍然响应
    cy.wait(2000)
    
    // 验证页面仍然可用
    cy.get('[data-testid="collector-list-container"]').should('be.visible')
    cy.get('[data-testid="collector-common-table"]').should('be.visible')
  })
})
