/// <reference types="cypress" />

describe('导航菜单功能测试', () => {
  beforeEach(() => {
    // 模拟登录状态
    cy.mockLogin()
    
    // 访问包含菜单的页面
    cy.visit('/#/dashboard')
  })

  it('应该显示侧边栏菜单', () => {
    // 检查菜单容器
    cy.get('[data-testid="sidebar-menu"]').should('be.visible')
    
    // 检查菜单是否有菜单项
    cy.get('[data-testid="sidebar-menu"] .el-menu-item, [data-testid="sidebar-menu"] .el-sub-menu').should('have.length.gt', 0)
  })

  it('应该能够点击菜单项进行导航', () => {
    // 获取所有菜单项
    cy.get('[data-testid^="menu-item-"]').then($items => {
      if ($items.length > 0) {
        // 点击第一个菜单项
        const firstItemTestId = $items[0].getAttribute('data-testid')
        cy.get(`[data-testid="${firstItemTestId}"]`).click()
        
        // 验证页面跳转
        cy.wait(1000)
        cy.url().should('not.be.empty')
      }
    })
  })

  it('应该支持子菜单展开和收起', () => {
    // 查找子菜单
    cy.get('[data-testid^="menu-submenu-"]').then($submenus => {
      if ($submenus.length > 0) {
        const firstSubmenuTestId = $submenus[0].getAttribute('data-testid')
        
        // 点击子菜单标题展开
        cy.get(`[data-testid="${firstSubmenuTestId}"] .el-sub-menu__title`).click()
        
        // 验证子菜单展开
        cy.get(`[data-testid="${firstSubmenuTestId}"]`).should('have.class', 'is-opened')
        
        // 再次点击收起
        cy.get(`[data-testid="${firstSubmenuTestId}"] .el-sub-menu__title`).click()
        
        // 验证子菜单收起
        cy.get(`[data-testid="${firstSubmenuTestId}"]`).should('not.have.class', 'is-opened')
      }
    })
  })

  it('应该高亮当前激活的菜单项', () => {
    // 检查当前URL对应的菜单项是否被激活
    cy.url().then(currentUrl => {
      const path = new URL(currentUrl).pathname.replace('/', '')
      
      if (path) {
        // 查找对应的菜单项
        cy.get(`[data-testid="menu-item-${path}"]`).then($item => {
          if ($item.length > 0) {
            // 验证菜单项有活跃状态
            cy.get(`[data-testid="menu-item-${path}"]`).should('have.class', 'is-active')
          }
        })
      }
    })
  })

  it('应该显示菜单图标', () => {
    // 检查菜单项是否有图标
    cy.get('[data-testid^="menu-item-"] .menu-icon, [data-testid^="menu-submenu-"] .menu-icon').should('have.length.gt', 0)
    
    // 验证图标是否正常显示
    cy.get('.menu-icon').each($icon => {
      cy.wrap($icon).should('be.visible')
      cy.wrap($icon).should('have.attr', 'src')
    })
  })

  it('应该支持菜单收起状态', () => {
    // 如果有收起按钮，测试收起功能
    cy.get('body').then($body => {
      // 查找可能的收起按钮（这里需要根据实际实现调整）
      if ($body.find('.hamburger, .menu-toggle').length > 0) {
        cy.get('.hamburger, .menu-toggle').click()
        
        // 验证菜单收起状态
        cy.get('[data-testid="sidebar-menu"]').should('have.class', 'el-menu--collapse')
        
        // 再次点击展开
        cy.get('.hamburger, .menu-toggle').click()
        cy.get('[data-testid="sidebar-menu"]').should('not.have.class', 'el-menu--collapse')
      }
    })
  })

  it('应该支持键盘导航', () => {
    // 使用Tab键在菜单项之间导航
    cy.get('[data-testid="sidebar-menu"]').focus()
    
    // 使用方向键导航
    cy.get('[data-testid="sidebar-menu"]').type('{downarrow}')
    cy.wait(500)
    cy.get('[data-testid="sidebar-menu"]').type('{uparrow}')
    
    // 使用Enter键选择
    cy.get('[data-testid="sidebar-menu"]').type('{enter}')
  })

  it('应该在不同页面显示正确的菜单状态', () => {
    // 导航到不同页面并验证菜单状态
    const testPages = ['dashboard', 'dust-overview']
    
    testPages.forEach(page => {
      cy.visit(`/${page}`)
      cy.waitForPageLoad()
      
      // 验证菜单仍然可见
      cy.get('[data-testid="sidebar-menu"]').should('be.visible')
      
      // 验证对应的菜单项被激活
      cy.get(`[data-testid="menu-item-${page}"]`).then($item => {
        if ($item.length > 0) {
          cy.get(`[data-testid="menu-item-${page}"]`).should('have.class', 'is-active')
        }
      })
    })
  })

  it('应该处理菜单权限控制', () => {
    // 验证菜单项是否根据权限显示
    cy.get('[data-testid^="menu-item-"], [data-testid^="menu-submenu-"]').each($item => {
      // 验证菜单项是可见的（说明用户有权限访问）
      cy.wrap($item).should('be.visible')
    })
  })

  it('应该在移动端正确显示', () => {
    // 测试移动端菜单
    cy.viewport(375, 667) // iPhone尺寸
    
    // 菜单应该仍然可见或者有移动端适配
    cy.get('[data-testid="sidebar-menu"]').should('exist')
    
    // 恢复桌面端
    cy.viewport(1280, 720)
    cy.get('[data-testid="sidebar-menu"]').should('be.visible')
  })

  it('应该处理菜单项的hover状态', () => {
    // 测试菜单项的鼠标悬停效果
    cy.get('[data-testid^="menu-item-"]').first().then($item => {
      if ($item.length > 0) {
        cy.wrap($item).trigger('mouseover')
        
        // 验证hover状态（这里需要根据实际CSS类名调整）
        cy.wrap($item).should('have.css', 'background-color')
        
        cy.wrap($item).trigger('mouseout')
      }
    })
  })
})