#!/bin/bash

# 通用测试执行脚本
# 用法: ./run-tests.sh [environment] [browser]
# 示例: ./run-tests.sh staging chrome

ENV=${1:-development}
BROWSER=${2:-chrome}

echo "开始执行 $ENV 环境的测试，使用 $BROWSER 浏览器"

# 使用固定配置，不依赖环境变量
echo "使用固定配置运行测试，不依赖环境变量"

# 检查是否在Windows上运行（通过环境变量或uname命令）
if [[ "$OSTYPE" == "msys"* ]] || [[ "$OSTYPE" == "win32" ]] || [[ "$(uname -s)" == MINGW* ]]; then
  echo "检测到Windows环境"
  echo "请使用 run-tests.bat 脚本在Windows上运行测试"
  echo "用法: run-tests.bat [environment] [browser]"
  exit 0
fi

# 检查浏览器是否可用
echo "检查浏览器 $BROWSER 是否可用..."
AVAILABLE_BROWSERS=$(npx cypress info | grep "Available browsers" -A 20 | grep -E "^\s*-" | sed 's/^\s*-\s*//' | tr '\n' ' ')
echo "系统可用的浏览器: $AVAILABLE_BROWSERS"

if [[ $AVAILABLE_BROWSERS != *"$BROWSER"* ]]; then
  echo "警告: 浏览器 $BROWSER 不可用，使用默认浏览器 electron"
  BROWSER="electron"
fi

# 执行测试 - 自动涵盖cypress/e2e下面的第一层测试脚本
echo "查找cypress/e2e目录下的第一层测试脚本..."
SPEC_FILES=""
for file in cypress/e2e/*.cy.js; do
  if [ -f "$file" ]; then
    if [ -z "$SPEC_FILES" ]; then
      SPEC_FILES="$file"
    else
      SPEC_FILES="$SPEC_FILES,$file"
    fi
    echo "  - $file"
  fi
done

# 确保报告目录存在
echo "创建报告目录..."
mkdir -p cypress/reports
mkdir -p public/reports

echo "清理旧的报告文件..."
# 清理旧的报告文件，但保留目录结构
find cypress/reports -name "*.json" -type f -delete 2>/dev/null || true
find cypress/reports -name "*.html" -type f -delete 2>/dev/null || true
find public/reports -name "*.json" -type f -delete 2>/dev/null || true
find public/reports -name "*.html" -type f -delete 2>/dev/null || true

if [ -n "$SPEC_FILES" ]; then
  echo "执行指定的测试脚本..."
  echo "运行命令: npx cypress run --browser $BROWSER --spec \"$SPEC_FILES\""
  npx cypress run --browser $BROWSER --spec "$SPEC_FILES"
else
  echo "未找到第一层测试脚本，执行所有测试..."
  npx cypress run --browser $BROWSER
fi

# 检查测试是否成功执行
TEST_EXIT_CODE=$?
echo "测试执行完成，退出码: $TEST_EXIT_CODE"

if [ $TEST_EXIT_CODE -eq 0 ]; then
  echo "测试执行成功，生成报告..."
else
  echo "警告: 测试执行失败（退出码: $TEST_EXIT_CODE），但仍尝试生成报告..."
fi

# 检查是否有JSON报告文件生成
JSON_REPORT_COUNT=$(find cypress/reports -name "*.json" -type f | wc -l)
echo "找到 $JSON_REPORT_COUNT 个JSON报告文件"

if [ $JSON_REPORT_COUNT -gt 0 ]; then
  echo "合并报告..."
  # 合并报告
  npx mochawesome-merge cypress/reports/**/*.json -o public/reports/merged-report.json
  MERGE_EXIT_CODE=$?
  if [ $MERGE_EXIT_CODE -ne 0 ]; then
    echo "报告合并失败，退出码: $MERGE_EXIT_CODE"
    exit 1
  fi
  
  # 生成HTML报告
  echo "生成HTML报告..."
  npx marge public/reports/merged-report.json --reportDir public/reports --inline
  MARGE_EXIT_CODE=$?
  if [ $MARGE_EXIT_CODE -ne 0 ]; then
    echo "HTML报告生成失败，退出码: $MARGE_EXIT_CODE"
    exit 1
  fi
  
  # 生成自定义汇总报告
  echo "生成自定义汇总报告..."
  node scripts/generate-summary-report.cjs
  SUMMARY_EXIT_CODE=$?
  if [ $SUMMARY_EXIT_CODE -ne 0 ]; then
    echo "自定义汇总报告生成失败，退出码: $SUMMARY_EXIT_CODE"
    exit 1
  fi
  
  echo "所有报告生成完成！"
  echo "查看汇总报告: public/reports/summary-report.html"
  echo "查看详细报告: public/reports/merged-report.html"
else
  echo "未找到任何JSON报告文件，跳过报告生成"
  if [ $TEST_EXIT_CODE -ne 0 ]; then
    echo "测试执行失败，请检查错误信息"
    exit 1
  fi
fi

echo "测试执行完成"